
#ifndef LEDA_INTEGER_MATRIX_H
#define LEDA_INTEGER_MATRIX_H

//------------------------------------------------------------------------------
//  matrices
//------------------------------------------------------------------------------

#include "numberdef"
#include <LEDA/basic.h>
#include <LEDA/array.h>
#include "integer_vector.h"


/*{\Manpage {integer\_matrix} {}  {Integer-Valued Matrices} }*/

class integer_matrix {

/*{\Mdefinition
An instance of the data type $integer\_matrix$ is a matrix of integer variables.  
}*/

  integer_vector** v;
  int  d1;
  int  d2;

  void     flip_rows(int,int);
  void     check_dimensions(const integer_matrix&) const; 
  number&  elem(int i, int j) const { return v[i]->v[j]; }
  number** triang(const integer_matrix&, int&) const;
    
public:

/*{\Mcreation M }*/

 integer_matrix(int n=0, int m=0);

/*{\Mcreate creates an instance $M$ of type $integer\_matrix$, $M$ is initialized to 
           the $n \times m$ - zero matrix. }*/

integer_matrix(const array<integer_vector> & A);

/*{\Mcreate creates an instance $M$ of type $integer\_matrix$. Let $A$
be an array of $m$ column-vectors of common dimension $n$. $M$ is initialized to an $n \times m$ matrix with the columns as specified by $A$. }*/



 

  integer_matrix(const integer_matrix&);
  integer_matrix(const integer_vector&);
  integer_matrix(int,int,number**);

  integer_matrix& operator=(const integer_matrix&);

 ~integer_matrix();

  LEDA_MEMORY(integer_matrix)


/*{\Moperations 3 5 }*/

int     dim1()  const  {  return d1; }
/*{\Mop  returns $n$, the number of rows of $M$. }*/

int     dim2()  const  {  return d2; }
/*{\Mop  returns $m$, the number of columns of $M$. }*/


integer_vector& row(int i) const
/*{\Mop  returns the $i$-th row of $M$ (an $m$-vector).
	 \precond  $0 \le i \le n-1$. }*/
{ 
#ifndef MY_OFF
if ( i<0 || i>=d1 )  error_handler(1,"matrix: row index out of range");
#endif
   return *v[i];
}

integer_vector  col(int i) const;
/*{\Mop  returns the $i$-th column of $M$ (an $n$-vector).
	 \precond  $0 \le i \le m-1$. }*/


operator integer_vector() const; 

integer_vector& operator[](int i)    const { return row(i); }

number& operator()(int i, int j)
/*{\Mfunop returns $M_{i,j}$. \\
	  \precond $0\le i\le n-1$ and $0\le j\le m-1$.}*/
{ 
#ifndef MY_OFF
if ( i<0 || i>=d1 )  error_handler(1,"matrix: row index out of range");
if ( j<0 || j>=d2 )  error_handler(1,"matrix: col index out of range");
#endif
  return elem(i,j);
}

number  operator()(int i,int j) const
{ 
#ifndef MY_OFF
if ( i<0 || i>=d1 )  error_handler(1,"matrix: row index out of range");
if ( j<0 || j>=d2 )  error_handler(1,"matrix: col index out of range");
#endif
  return elem(i,j);
}

int     operator==(const integer_matrix&)    const;
int     operator!=(const integer_matrix& x)  const { return !(*this == x); }

integer_matrix operator+(const integer_matrix& M1);
/*{\Mbinop Addition. \\
 	   \precond $M$.dim1() = $M1$.dim1() and $M$.dim2() = $M1$.dim2().}*/

integer_matrix operator-(const integer_matrix& M1);
/*{\Mbinop Subtraction. \\
	   \precond $M$.dim1() = $M1$.dim1() and
	   $M$.dim2() = $M1$.dim2().}*/

integer_matrix operator-(); // unary

integer_matrix& operator-=(const integer_matrix&);
integer_matrix& operator+=(const integer_matrix&);

integer_matrix operator*(const integer_matrix& M1) const;
/*{\Mbinop Multiplication. \\
	   \precond $M$.dim2() = $M1$.dim1().}*/

integer_vector operator*(const integer_vector& vec) const
{ return integer_vector(*this * integer_matrix(vec)); }
/*{\Mbinop  Multiplication with vector.\\
	    \precond $M$.dim2() = $vec$.dim(). }*/

integer_matrix operator*(number x) const;
/*{\Mbinop Multiplication with number |x|.}*/

friend integer_matrix identity(int n);
/*{\Mfunc  returns  an $n$ by $n$ identity matrix. }*/

friend integer_matrix  trans(const integer_matrix& M);
/*{\Mfunc  returns  $M^T$ ($m\times n$ - matrix). }*/


friend bool  inverse(const integer_matrix& M, 
integer_matrix & inverse , number & D,integer_vector & c);
/*{\Mfuncl  determines whether $M$ has an inverse. It also computes either
the inverse as $(1/D) \cdot inverse$ or a vector $c$ such that 
$c^T \cdot M = 0 $.  }*/


friend integer_matrix  inverse(const integer_matrix& M, number & D)
/*{\Mfuncl  returns the inverse matrix of $M$. More precisely, $1/D$ times the
matrix returned is the inverse of $M$.\\
	 \precond  $M$.det() $\neq$ 0. }*/
{integer_matrix result;
integer_vector c;
if (inverse(M,result,D,c)) return result;
error_handler(1,"inverse: matrix is singular");
}

friend number  determinant(const integer_matrix& M, 
integer_matrix & L, integer_matrix & U, array<int> & q,
integer_vector & c);
/*{\Mfuncl  returns the determinant $D$ of $M$ and sufficient information to verify that the value of the determinant is correct. If the determinant is zero then
$c$ is a vector such that $c^T \cdot A = 0$. If the determinant is non-zero then $L$ and $U$ are lower and upper diagonal matrices respectively, and $q$ encodes a permutation matrix $Q$ with $Q(i,j) = 1$ iff $i = q(j)$ 
such that $L \cdot A \cdot Q = U$,
$L(0,0) = 1$,
$L(i,i) = U(i-1,i-1)$ for all $i$, $1 \le i < n$, and 
$D = s \cdot U(n-1,n-1)$ where $s$ is the determinant of $Q$. \\
 \precond  $M$ is quadratic. }*/

friend bool verify_determinant(const integer_matrix& M, number D,
integer_matrix & L, integer_matrix & U, array<int> q,
integer_vector & c);
/*{\Mfuncl verifies the conditions stated above. }*/


friend number  determinant(const integer_matrix& M);
/*{\Mfuncl  returns the determinant of $M$.\\
	 \precond  $M$ is quadratic. }*/




friend bool linear_solver(const integer_matrix & M, const integer_vector& b, integer_vector &x, number & D, integer_matrix & spanning_vectors, integer_vector & c); 
/*{\Mfuncl determines the complete solution space of the linear system $M\cdot x = b$. If the system is unsolvable then $c^T \cdot M = 0$ and $c^T \cdot b \not= 0$. If the system is solvable then $(1/D) x$ is a solution, and the columns of
$spanning\_vectors$ are a maximal set of linearly independent solutions to the
corresponding homogeneous system.\\
	 \precond $M$.dim1() = $b$.dim(). }*/

friend bool linear_solver(const integer_matrix & M, const integer_vector& b, integer_vector &x, number & D, integer_vector & c) 
/*{\Mfuncl determines whether the linear system $M\cdot x = b$ is solvable. If
yes, then $(1/D) x$ is a solution, if not then $c^T \cdot M = 0$ and $c^T \cdot b \not= 0$.\\
	 \precond $M$.dim1() = $b$.dim(). }*/
{
integer_matrix spanning_vectors;
return linear_solver(M,b,x,D,spanning_vectors,c);
}

friend bool linear_solver(const integer_matrix & M, const integer_vector& b, integer_vector &x, number & D) 
/*{\Mfuncl as above, but without the witness $c$ \\
	 \precond $M$.dim1() = $b$.dim(). }*/
{
integer_matrix spanning_vectors;
integer_vector c;
return linear_solver(M,b,x,D,spanning_vectors,c);
}

friend bool solvable(const integer_matrix & M, const integer_vector& b) 
/*{\Mfuncl determines whether the system $M \cdot x = b$ is solvable \\
	 \precond $M$.dim1() = $b$.dim(). }*/
{
integer_matrix spanning_vectors;
integer_vector c;
integer_vector x;
number D;
return linear_solver(M,b,x,D,spanning_vectors,c);
}


friend bool homogeneous_linear_solver(const integer_matrix & M, integer_vector &x); 
/*{\Mfuncl determines whether the homogeneous linear system $M\cdot x = 0$ has a non-trivial solution. If
yes, then $x$ is such a solution.}*/

friend int rank(const integer_matrix & M); 
/*{\Mfuncl returns the rank of the matrix $M$}*/






friend ostream& operator<<(ostream& O, const integer_matrix& M);
/*{\Mbinopfunc  writes matrix $M$ row by row to the output stream $O$. }*/

friend istream& operator>>(istream& I, integer_matrix& M);
/*{\Mbinopfunc  reads matrix $M$ row by row from the input stream $I$. }*/

/*{\Mimplementation
The data type $integer\_matrix$ is implemented by two-dimensional arrays of integers. 
Operations det, solve, and inv take time $O(n^3)$, dim1, dim2, row, and col 
take constant time, all other operations take time $O(nm)$.  
The space requirement is $O(nm)$.

All functions on integer matrices compute the exact result, i.e., there is no
rounding error. The implemenation follows a 
proposal of J. Edmonds (J. Edmonds, Systems of distinct representatives and
linear algebra, Journal of Research of the Bureau of National 
Standards, (B), 71, 241 -245). Most functions of linear algebra are
{\em checkable}, i.e., the programs can be asked for a 
proof that their output is correct. For example, if the linear system solver 
declares a linear system $A x = b$ unsolvable it also returns a vector $c$
such that $c^T A = 0$ and $c^T b \not= 0$.
}*/

};


inline void Print(const integer_matrix& m, ostream& out) { out << m; }
inline void Read(integer_matrix& m, istream& in)         { in >> m;  }

inline int compare(const integer_matrix&, const integer_matrix&) 
{ error_handler(1,"compare not defined for type `matrix`"); 
  return 0;
 }


LEDA_TYPE_PARAMETER(integer_matrix)

#endif
