#ifndef HYPERPLANE_H
#define HYPERPLANE_H


class d_rat_point;
class d_rat_segment;
class hyperplane;
int compare_help(const hyperplane &p, const hyperplane &q);

inline int compare(const hyperplane &p, const hyperplane &q)
{return compare_help(p,q);}


#include "numberdef"
#include "d_rat_point.h"
#include <LEDA/array.h>
#include <LEDA/integer.h>
#include <LEDA/handle_types.h>


 


//------------------------------------------------------------------------------
// hyperplanes
//------------------------------------------------------------------------------

class hyperplane_rep  : public handle_rep {

friend class d_rat_point;
friend class d_rat_segment;
friend class hyperplane;

int dim;   
number* coefficients;


/* The class hyperplane is a handle class. I design is analogously to
the class d_rat_point. In fact the internal representation is the same.
A hyperplane in d dimensions is represented by an array 
|coefficients[d+1]|. The plane equation is

$$c[0] + sum (c[i]*x[i]/x[0])  = 0.$$

I do not make any assumption about the sign of |x[0]| or |c[0]|. Of course,
|c[0]| may be zero, but |x[0]| may not be zero. */



public:

   
hyperplane_rep(int d = 0)  // d = 0 is default constructor
   { dim = d;
     coefficients = new number[d+1];
     for (int i = 1; i <= dim; i++) coefficients[i] = 0;
    }


   // hyperplane_rep() {hyperplane_rep(0); }  //contained in above


hyperplane_rep(const integer_vector& c) 
    { dim = c.dim()-1;
      coefficients = new number[dim+1];
      for (int i = 0; i <= dim; i++) coefficients[i] = c[i];
    }

hyperplane_rep(const array<d_rat_point>&  P, const d_rat_point & o, int side);

hyperplane_rep(const array<d_rat_point>&  P, 
const array<integer_vector> &  N, const d_rat_point & o, int side);

~hyperplane_rep() {delete[] coefficients;}





   
LEDA_MEMORY(hyperplane_rep)
};



/*{\Manpage {hyperplane} {} {Arbitrary-Dimensional Hyperplanes} }*/

class hyperplane  : public handle_base {

/*{\Mdefinition
An instance of the data type $hyperplane$ is a hyperplane  with rational coefficients 
in an arbitrary dimensional space. A hyperplane $h$ is represented by 
coefficients $(h_0,h_1,\ldots,h_d)$ of arbitrary length integers. The plane equation is $h[0] + \sum h[i]*x[i]/x[0] = 0$,
where $x[0]$ to $x[d]$ are homogeneous point coordinates. The sign of the left hand side of this 
expression determines the position of a point $x$ with respect to the
hyperplane (on the hyperplane, negative side, or positive side). The normal
vector of the hyperplane is $(h[1], \ldots, h[d])$}*/



hyperplane_rep* ptr() const { return (hyperplane_rep*)PTR; } 

public:



/*{\Mcreation h}*/

hyperplane() { PTR = new hyperplane_rep; }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to a hyperplane in 0-dimensional space.}*/



hyperplane(int d)       { PTR = new hyperplane_rep(d); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to a hyperplane in $d$-dimensional space.}*/

hyperplane(const integer_vector& c){ PTR = new hyperplane_rep(c); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the hyperplane with coefficients $c$. }*/

hyperplane(const array<d_rat_point>&  P, const array<integer_vector> &  N, const d_rat_point & o, int side){PTR = new hyperplane_rep(P,N,o,side);}
/*{\Mcreate constructs a hyperplane that passes through the points in $P$, 
whose normal 
is a linear combination of the vector in $N$, and which has the 
point $o$ on $side$, where $side$ is in $\{-1,0,+1\}$. $P$ and $N$ must both
be non-empty } */


hyperplane(const array<d_rat_point>&  P, const d_rat_point & o, int side)
    {PTR = new hyperplane_rep(P,o,side);}
/*{\Mcreate constructs a hyperplane that passes through the points in $P$, 
and which has the 
point $o$ on $side$, where $side$ is in $\{-1,0,+1\}$. $P$ must consist of
$d$ points where $d$ is the dimension of the space. } */



 hyperplane(const hyperplane& p) : handle_base(p) {}
~hyperplane()    {}       //       { clear(); } relict

 hyperplane& operator=(const hyperplane& p) 
 { handle_base::operator=(p); return *this; }

/*{\Moperations 2 3.5 }*/


number operator[](int i)const {return ptr()->coefficients[i];}
/*{\Marrop     returns the $i$-th coefficient of \var.}*/

int dim() const {return ptr()->dim;}
/*{\Mop     returns the dimension of \var.}*/


integer_vector equation() const;
/*{\Mop returns the equation of \var.}*/

integer_vector normal() const;
/*{\Mop returns the normal vector of \var. It points from the negative halfspace into the positive halfspace and its entries are $(c1, \ldots, cd)$. }*/

number value_at(d_rat_point p) const;
/*{\Mop returns the value of \var at the point |p|,i.e., 
$\sum_{0 \le i \le d} h_ip_i$}\\
{\em warning:} this has to be divided by |p[0]| to get the true value.}*/

hyperplane reverse() const {return hyperplane(-equation());}
/*{\Mop returns a hyperplane with the same equation
$which_side()$ of this hyperplane returns the complementary value for each point. (0
remains 0, of course) }*/




friend int which_side( const hyperplane & h, const d_rat_point & x);
    /*{\Mfuncl decides on which side of the hyperplane $h$ the
point $x$ lies }*/





friend bool identical(const hyperplane& p, const hyperplane& q)
{ return p.ptr() == q.ptr(); }
/*{\Mbinopfunc  test for identity.}*/

friend bool operator==(const hyperplane& p, const hyperplane& q);
/*{\Mbinopfunc  test for equality.}*/

friend bool operator!=(const hyperplane& p, const hyperplane& q);
/*{\Mbinopfunc  test for inequality.}*/

friend ostream& operator<<(ostream& O, const hyperplane& p) ;
/*{\Mbinopfunc  writes the coefficients  of \var\ to 
                output stream $O$.}*/

friend istream& operator>>(istream& I, hyperplane& p) ;
/*{\Mbinopfunc  reads the coefficients of \var\ from 
                input stream $I$. This operator uses the current dimension
                of  \var.}*/




//friend int compare_help(const hyperplane&, const hyperplane&);

int cmp(const hyperplane&, const hyperplane&);  //hier stand eine static davor

};

//LEDA_HANDLE_TYPE(hyperplane)

inline void Print(const hyperplane& p, ostream& out) { out << p; } 
inline void Read(hyperplane& p,  istream& in)        { in >> p; }



inline bool operator==(const hyperplane& p, const hyperplane& q)
{ return compare(p,q) == 0; }

inline bool operator!=(const hyperplane& p, const hyperplane& q)
{ return compare(p,q) != 0; }








#endif





