/* (c) 2005 Joerg Dorchain <joerg@dorchain.net>
 *
 * takes some inspiration and code from shout2rtp by Warren Toomey, wkt@tuhs.org
 *
 * released under GPL version2 or later
 */

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <db.h>

#include "globals.h"

void ResetTime(void)
{
  struct timeval sTime;

  gettimeofday(&sTime, NULL);
  iBaseTime = sTime.tv_sec;
}

long long GetTime_ll(void)
{
  struct timeval sTime;

  gettimeofday(&sTime, NULL);
  if (iBaseTime == 0)
    iBaseTime = sTime.tv_sec;
  return (((long long)sTime.tv_sec - (long long)iBaseTime) * 100000) + ((long long)sTime.tv_usec / 10);
}


/* like read, 
 * but tries get get maximal amount of data
 * and never return EAGAIN
 */
int myread(int fd, void *buf, size_t count)
{
  size_t rb;
  size_t br = 0;

  while (count > 0) {
    if ((rb = read(fd, buf, count)) < 0) {
      if (errno != EAGAIN)
	return rb;
    } else {
       if (rb == 0)
	 return br;
       br += rb;
       count -= rb;
       buf += rb;
    }
  }
  return br;
}

/* Initializes rtpsocket and rtptoaddr
 * return 0 on success
 * else errno ist set
 */
int makesocket(void)
{
  int ret;
  const int one = 1;

  if ((rtpsocket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0 )
    return rtpsocket;
  if ((ret = setsockopt(rtpsocket, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one))))
    return ret;
  if ((ret = setsockopt(rtpsocket, IPPROTO_IP, IP_MULTICAST_TTL, &multicasttl, sizeof(multicasttl))))
    return ret;
  rtptoaddr.sin_family = AF_INET;
  rtptoaddr.sin_port = htons(multicastport);
  if (!(ret = inet_aton(multicastaddr, &rtptoaddr.sin_addr)))
    return !ret;
  return 0;
}

/*
 * Read and decode the first 3 octets from an MP3 frame to work out frame
 * size, bitrate and samplerate. If dontread==1, use the previous values.
 * Returns 1 if ok, 0 for EOF, < 0 if an error. Copy the 3 bytes into pBuffer if ok.
 */
int GetMPEGFrameInfo_b(unsigned char *pBuffer, int dontread, int *iFrameSize, int *iBitrate, int *iSamplerate)
{
static const int bitrate_table[2][4][16] = {
  /* MPEG II */
  {
    /* reserved */
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* layer 3 */
    {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
    /* layer 2 */
    {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
    /* layer 1 */
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 356, 0}
  },
  /* MPEG I */
  {
    /* reserved */
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* layer 3 */
    {0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0},
    /* layer 2 */
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0},
    /* layer 1 */
    {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0}
  }
};

static const int samplerate_table[2][3] = {
  {22050, 24000, 16000},
  {44100, 48000, 32000}
};

static unsigned char fheader[3];

  int err;
  int mpeg;
  int layer;
  int bitrate_index;
  int samplerate_index;
  int pad;
  int no_valid_header=1;

  while (no_valid_header) {
    if (!dontread) {
      fheader[0]= 0;
      while (1) {
        if ((err = myread(currentfile, &fheader[1], 1)) != 1) {
	  if (err != 0)
	    dbgp(1, "readerror while no valid header: %s\n", strerror(errno));
	  return err;
	}
        if (fheader[0] == 0xff && (fheader[1] & 0xf0) == 0xf0)
	  break;
        fheader[0] = fheader[1];
      }
      if ((err = myread(currentfile, &fheader[2], 1)) != 1)
	return err;
    }
    mpeg= (fheader[1] & 0x08) >> 3;
    layer= (fheader[1] & 0x06) >> 1;
    bitrate_index= (fheader[2] & 0xf0) >> 4;
    samplerate_index= (fheader[2] & 0x0c) >> 2;
    pad= (fheader[2] & 0x02) >> 1;

    *iBitrate= bitrate_table[mpeg][layer][bitrate_index];
    *iSamplerate= samplerate_table[mpeg][samplerate_index];
  
    /* Exit the outside loop if we now have a valid header */
    if ((*iBitrate != 0) && (*iSamplerate != 0)) break;
    dbgp(1, "Frame error: mpeg %d layer %d bit %d sample %d pad %d\n",
      mpeg, layer, bitrate_index, samplerate_index, pad);
    dontread=0; 
  }

  *iFrameSize= (144000 * *iBitrate / *iSamplerate) + pad;

  pBuffer[0]= fheader[0]; pBuffer[1]= fheader[1]; pBuffer[2]= fheader[2];
  return 1;
}

/*
 * Construct a packet by reading in as many MP3 frames as we can until there
 * is no more room in the buf. Return the number of frames read in. If we
 * return 0, then we have either reached end of file or the packet size is
 * too small for even a single frame.
 */
int MakePacket(unsigned char *buf, const int pktsize, int *iBytesInPacket,
	   int *iSampleRate)
{
  static int leftover_frame = 0;
  int iFrameSize, iNumFrames= 0, iBitRate;
  int err;
  int numread;

  *iBytesInPacket= 0;
  while (1) {
    if ((err = GetMPEGFrameInfo_b(buf, leftover_frame, &iFrameSize, &iBitRate, iSampleRate)) != 1) {
      leftover_frame = 0;
      return err;
    }
    if ((*iBytesInPacket + iFrameSize) > pktsize) {
      leftover_frame = 1;
      dbgp(2, "Can't fit %d and %d into %d\n",*iBytesInPacket, iFrameSize, pktsize);
      return iNumFrames;
    }
    /* We have enough room to read in the rest of the frame */
    leftover_frame = 0;
    (*iBytesInPacket) += iFrameSize;
    buf += 3; iFrameSize -= 3;
    while (iFrameSize > 0) {
      if ((numread = myread(currentfile, buf, iFrameSize)) <= 0 ) {
	leftover_frame = 0;
	if (numread < 0)
	  dbgp(1, "readerror: %s\n", strerror(errno));
	return numread;
      }
      iFrameSize -= numread; buf += numread;
    }
    iNumFrames++;
  }
}

/* sends out the currentfile
 * via rtpsocket to rtptoaddr
 * return 0 on success
 */
int sendframes(void)
{
  int ret;
  unsigned char pbuffer[PACKETSIZE];
  struct rtpheader *ppacket;
  int iFlagMask;
  int packet_delay, frames_in_pkt;
  int iBytesCopied, iSampleRate;
  unsigned int uPacketNum, uSequenceNumber, time_between_packets;
  long long llPacketTime= 0, llCurrentTime;
  struct timespec sleeptime;
  int hdrlen = sizeof(struct rtpheader);

  ppacket = (struct rtpheader *)pbuffer;
  iFlagMask = 0x800E << 16;
  ppacket->syncsource_id = htonl(rand());
  ppacket->audioheader = 0;
  sleeptime.tv_sec = 0;

  for (uPacketNum=0, uSequenceNumber=rand(); /* nothing */ ; uPacketNum++, uSequenceNumber++) {
    if ((frames_in_pkt = MakePacket(&pbuffer[hdrlen],
	          sizeof(pbuffer) - hdrlen, &iBytesCopied, &iSampleRate)) <= 0 )
      return frames_in_pkt;
    dbgp(2, "Makepacket %d\n", frames_in_pkt);
    ppacket->flags= htonl(iFlagMask | (uSequenceNumber & 0xFFFF));
    /* iTimeBetweenPackets is in 10us units */
    time_between_packets = ((1000 * 1152) / (iSampleRate / 100)) * frames_in_pkt;
    /* uPacketTime is in 10us units */
    llPacketTime += (long long)time_between_packets;
    ppacket->timestamp = 0;
    if (uPacketNum == 0)
      ResetTime();
    llCurrentTime = GetTime_ll();
    packet_delay = (int)(llPacketTime - llCurrentTime);
    if (packet_delay < 0)
      packet_delay = 0;
    sleeptime.tv_nsec = packet_delay * 10000;
    nanosleep(&sleeptime, NULL);
    if ((ret = sendto(rtpsocket, pbuffer, iBytesCopied + hdrlen, 0,
	                 (struct sockaddr *)&rtptoaddr, sizeof(rtptoaddr))) < 0)
      return ret;
  }
}

/* Play file pointed to by fn
 * return 0 on success
 * else errno ist set
 */
int playerthread(char *fn)
{
  int ret;
  FILE *pidfile;

  if ((pidfile = fopen(pidfilename, "w")) == NULL)
     return errno;
  if (fprintf(pidfile, "%d\n", getpid()) < 0) {
     ret = errno;
     fclose(pidfile);
     return ret;
  }
  if (fclose(pidfile))
     return errno;
  if ((currentfile = open(fn, O_RDONLY)) < 0)
    return currentfile;
  if ((ret = sendframes()))
    return ret;
  if ((ret = close(currentfile)) < 0)
    return ret;
  currentfile = -1;
  if (unlink(pidfilename))
    return errno;
  return 0;
}

int playonefile(char *fn)
{
  pid_t player;
  int ret;
  int status;

  switch ((player = fork())) {
    case -1: /* error */
      return -1;
      break;
    case 0: /* child */
      exit(playerthread(fn));
      break;
  }
  if ((ret = waitpid(player, &status, 0)) != player)
    return ret;
  return WEXITSTATUS(status);
}

int playlistcallback(struct pl_state* state, DB * dbp, DBT *key, DBT *value)
{
  int weight;

  if ((playlist[state->i].name = strdup(key->data)) == NULL)
    return -1;
  playlist[state->i].weighted_pos = state->max_weight;
  dbgp(2, "Playlist entry %d name %s weith %d\n", state->i, playlist[state->i].name, playlist[state->i].weighted_pos);
  weight = atoi(value->data);
  if (weight < 1)
    weight = 1;
  state->max_weight += weight;
  state->i++;
  return 0;
}

/* reads in playlist fn
 * assumes no changes while playlist is open
 */
int init_playlist(char *fn)
{
  static time_t lastcheck = 0;
  int ret;
  int i;
  struct stat statbuf;
  DB *dbp;
  DB_HASH_STAT *dbstat;
  struct pl_state state;

  /* re-read only when modified since last time */
  if ((ret = stat(fn, &statbuf)))
    return ret;
  if (statbuf.st_mtime <= lastcheck)
    return 0;
  lastcheck = statbuf.st_mtime;
  for (i = 0; i < nr_entries; i++)
    free(playlist[i].name);
  free(playlist);
  playlist = NULL;
  nr_entries = 0;
  /* fill in playlist */
  if ((ret = db_create(&dbp, NULL, 0)) != 0)
    return ret;
  if ((ret = dbp->open(dbp, NULL, fn, NULL, DB_HASH, DB_UPGRADE, 0)) != 0)
    return ret;
  if ((ret = dbp->stat(dbp, &dbstat, 0)) != 0) {
    dbp->close(dbp, 0);
    return ret;
  }
  nr_entries = dbstat->hash_ndata;
  free(dbstat);
  dbgp(2, "playlist has %d lines\n", nr_entries);
  if ((playlist = calloc(nr_entries+1, sizeof(struct playlistentry))) == NULL ) {
    dbp->close(dbp, 0);
    return -1;
  }
  state.i = 0;
  state.max_weight = 0;
  if ((ret = iterate_db(dbp, playlistcallback, &state)) != 0)
    return ret;
  playlist[nr_entries].name = NULL;
  playlist[nr_entries].weighted_pos = state.max_weight;
  dbgp(2, "Last entry %d weith %d\n", nr_entries, state.max_weight);
  return 0;
}

int decreasecallback(void *dummy, DB * dbp, DBT *key, DBT *value)
{
  int weight, a;
  char linebuf[LINESIZE];

  weight = atoi(value->data);
  if (weight < 1)
    weight = 1;
  a = weight * expire_pct / 100;
  if (a < 1)
    a = 1;
  weight -= a;
  if (weight < 1)
    weight = 1;
  if ((a = snprintf(linebuf, LINESIZE, "%d", weight)) > LINESIZE)
    a = LINESIZE;
  value->size = a+1;
  value->data = linebuf;
  return dbp->put(dbp, NULL, key, value, 0);
}

/* decreases votes logarithmically
 * with additional restrictions
 * decreases is at least 1, no result < 1
 * works on files
 */
int decrease_votes(void)
{
  static time_t last_check = 0;
  time_t now;
  int ret;
  DB *dbp;

  if ((now = time(NULL)) < last_check + expire_per )
    return 0;
  last_check = now;
  if ((ret = db_create(&dbp, NULL, 0)) != 0)
    return ret;
  if ((ret = dbp->open(dbp, NULL, playlistfile, NULL, DB_HASH, DB_UPGRADE, 0)) != 0)
    return ret;
  if ((ret = iterate_db(dbp, decreasecallback, NULL)) != 0)
    return ret;
  return 0;
}

char * findsong(int wpos)
{
  int dir, i, step;

  dbgp(2, "wpos %d\n", wpos);
  i = nr_entries/2;
  step = i;
  while(!((playlist[i].weighted_pos <= wpos) && (playlist[i+1].weighted_pos > wpos))) {
    dir = (playlist[i].weighted_pos < wpos)?1:-1;
    step/=2;
    if (step < 1)
      step = 1;
    i+=dir*step;
    dbgp(2, "i %d step %d dir %d name %s pos %d\n", i, step, dir, playlist[i].name, playlist[i].weighted_pos);
  }
  return playlist[i].name;
}

char * next_static_song()
{
  static char ret[LINESIZE];
  FILE *new, *old;
  char linebuf[LINESIZE];
  char tmpnametmpl[] = "newstatic.XXXXXX";
  char tmpname[] = "newstatic.XXXXXX";
  int a;

  if ((old = fopen(staticfile, "r")) == NULL)
    return old;
  strcpy(tmpname, tmpnametmpl);
  if ((a = mkstemp(tmpname)) < 0)  {
    fclose(old);
    return NULL;
  }
  if ((new = fdopen(a, "w")) == NULL) {
    fclose(old);
    return new;
  }    
  if (fgets(ret, LINESIZE, old) == NULL)
     goto out;
  while (fgets(linebuf, LINESIZE, old) != NULL) {
    if (fputs(linebuf, new) == EOF)
      goto out;
  }
  if (fclose(new) != 0) {
    fclose(old);
    return NULL;
  }
  if (fclose(old) != 0)
    return NULL;
  if (rename(tmpname, staticfile) != 0)
    return NULL;
  ret[strlen(ret)-1] = 0;
  return ret;

out:
  a = errno;
  fclose(new);
  fclose(old);
  unlink(tmpname);
  errno = a;
  return NULL;
}

char * findnextsong(void)
{
  char *ret;

  if ((ret = next_static_song()))
    return ret;
  if (init_playlist(playlistfile))
    return NULL;
  return findsong(rand()%playlist[nr_entries].weighted_pos);
}

/* make fn first line in recent file
 * recentfile is shortened to RECENTLINES
 */
int updaterecentfile(char *fn)
{
  char tmpnametmpl[] = "recntpls.XXXXXX";
  char tmpname[] = "recntpls.XXXXXX";
  int i,a;
  FILE *new, *old;
  char linebuf[LINESIZE];

  if ((old = fopen(recentlyplayedfile, "r")) == NULL)
    return -1;
  strcpy(tmpname, tmpnametmpl);
  if ((a = mkstemp(tmpname)) < 0)  {
    fclose(old);
    return a;
  }
  if ((new = fdopen(a, "w")) == NULL) {
    fclose(old);
    return -1;
  }    
  if (fprintf(new, "%s\n", fn) < 0)
    goto out;
  for (i = 0; (i < RECENTLINES) && (fgets(linebuf, LINESIZE, old) != NULL); i++) {
    if (fputs(linebuf, new) == EOF)
      goto out;
  }
  if (fclose(new) != 0) {
    fclose(old);
    return -1;
  }
  if (fclose(old) != 0)
    return -1;
  if ((a = rename(tmpname, recentlyplayedfile)) != 0)
    return a;
  return 0;

out:
  a = errno;
  fclose(new);
  fclose(old);
  unlink(tmpname);
  errno = a;
  return -1;
}

int main(int argc, char **argv)
{
  char *filename;
  /* parse config */
  /* set up socket */
  if (makesocket()) {
    perror("makesocket");
    exit(errno);
  }
  /* setup stuff */
  srand(getpid()); /* inititalize random generator */
  /* main loop */
  while (1) {
    /* decrease votes */
    if (decrease_votes()) {
      perror("decreasevotes");
      exit(errno);
    }
    /* find next song */
    if ((filename = findnextsong()) == NULL) {
      perror("findnextsong");
      exit(errno);
    }
    if (updaterecentfile(filename)) {
      perror("updaterecentfile");
      exit(errno);
    }
    /* play it (see shout2rtp) */
    dbgp(1, "Now Playing: %s\n", filename);
    if (playonefile(filename)) {
      perror("playonefile");
      exit(errno);
    }
  }
  return 0; /* not reached */
}
